#!/usr/bin/env npx ts-node

/**
 * Dynamic Sitemap Generator for Programmatic SEO
 *
 * Usage:
 *   npx ts-node generate-sitemap.ts --base-url https://example.com --output public/sitemap.xml
 *   npx ts-node generate-sitemap.ts --base-url https://example.com --input data.json
 */

import * as fs from 'fs'
import * as path from 'path'

// Types
interface SitemapEntry {
  loc: string
  lastmod?: string
  changefreq?: 'always' | 'hourly' | 'daily' | 'weekly' | 'monthly' | 'yearly' | 'never'
  priority?: number
}

interface PageData {
  slug: string
  lastUpdated?: string
  priority?: number
}

// Utility functions
function escapeXml(str: string): string {
  return str
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&apos;')
}

function formatDate(date?: string | Date): string {
  const d = date ? new Date(date) : new Date()
  return d.toISOString().split('T')[0]
}

function generateSitemapXml(entries: SitemapEntry[]): string {
  const urlEntries = entries.map(entry => {
    let url = `  <url>\n    <loc>${escapeXml(entry.loc)}</loc>\n`
    if (entry.lastmod) {
      url += `    <lastmod>${entry.lastmod}</lastmod>\n`
    }
    if (entry.changefreq) {
      url += `    <changefreq>${entry.changefreq}</changefreq>\n`
    }
    if (entry.priority !== undefined) {
      url += `    <priority>${entry.priority.toFixed(1)}</priority>\n`
    }
    url += '  </url>'
    return url
  }).join('\n')

  return `<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
${urlEntries}
</urlset>`
}

// Scan app directory for pages
function scanAppDirectory(appDir: string): string[] {
  const pages: string[] = []

  function scan(dir: string, basePath: string = '') {
    if (!fs.existsSync(dir)) return

    const entries = fs.readdirSync(dir, { withFileTypes: true })

    for (const entry of entries) {
      const fullPath = path.join(dir, entry.name)

      if (entry.isDirectory()) {
        // Skip special directories
        if (entry.name.startsWith('_') || entry.name === 'api') continue

        const newBasePath = entry.name.startsWith('[')
          ? basePath // Dynamic routes handled separately
          : path.join(basePath, entry.name)

        scan(fullPath, newBasePath)
      } else if (entry.name === 'page.tsx' || entry.name === 'page.ts') {
        if (basePath) pages.push(basePath)
        else pages.push('/')
      }
    }
  }

  scan(appDir)
  return pages
}

// Load data from JSON file
function loadDataFile(filePath: string): PageData[] {
  const content = fs.readFileSync(filePath, 'utf-8')
  const parsed = JSON.parse(content)
  return Array.isArray(parsed) ? parsed : parsed.data || parsed.items || []
}

// Main function
async function main() {
  const args = process.argv.slice(2)
  const baseUrlIndex = args.indexOf('--base-url')
  const outputIndex = args.indexOf('--output')
  const inputIndex = args.indexOf('--input')
  const appDirIndex = args.indexOf('--app-dir')

  if (baseUrlIndex === -1) {
    console.error('Usage: npx ts-node generate-sitemap.ts --base-url <url> [--output <file>] [--input <data.json>] [--app-dir <path>]')
    process.exit(1)
  }

  const baseUrl = args[baseUrlIndex + 1].replace(/\/$/, '')
  const outputFile = outputIndex !== -1 ? args[outputIndex + 1] : 'public/sitemap.xml'
  const inputFile = inputIndex !== -1 ? args[inputIndex + 1] : null
  const appDir = appDirIndex !== -1 ? args[appDirIndex + 1] : 'app'

  const entries: SitemapEntry[] = []
  const today = formatDate()

  // Add homepage
  entries.push({
    loc: baseUrl,
    lastmod: today,
    changefreq: 'daily',
    priority: 1.0
  })

  // Scan app directory for static pages
  console.log(`\n📁 Scanning ${appDir} for pages...`)
  const staticPages = scanAppDirectory(appDir)

  for (const page of staticPages) {
    if (page === '/') continue // Already added
    entries.push({
      loc: `${baseUrl}${page.startsWith('/') ? page : '/' + page}`,
      lastmod: today,
      changefreq: 'weekly',
      priority: 0.8
    })
  }
  console.log(`   Found ${staticPages.length} static pages`)

  // Load dynamic pages from data file
  if (inputFile) {
    console.log(`📊 Loading data from ${inputFile}...`)
    const data = loadDataFile(inputFile)

    for (const item of data) {
      entries.push({
        loc: `${baseUrl}/${item.slug}`,
        lastmod: item.lastUpdated ? formatDate(item.lastUpdated) : today,
        changefreq: 'monthly',
        priority: item.priority || 0.6
      })
    }
    console.log(`   Added ${data.length} dynamic pages`)
  }

  // Generate sitemap
  const sitemap = generateSitemapXml(entries)

  // Ensure output directory exists
  const outputDir = path.dirname(outputFile)
  if (!fs.existsSync(outputDir)) {
    fs.mkdirSync(outputDir, { recursive: true })
  }

  fs.writeFileSync(outputFile, sitemap)
  console.log(`\n✅ Sitemap generated: ${outputFile}`)
  console.log(`   Total URLs: ${entries.length}`)

  // Also generate Next.js sitemap.ts if requested
  if (args.includes('--nextjs')) {
    generateNextjsSitemap(appDir, baseUrl, entries)
  }
}

function generateNextjsSitemap(appDir: string, baseUrl: string, entries: SitemapEntry[]) {
  const content = `import type { MetadataRoute } from 'next'

export default function sitemap(): MetadataRoute.Sitemap {
  const baseUrl = '${baseUrl}'

  return [
${entries.map(e => `    {
      url: '${e.loc}',
      lastModified: new Date('${e.lastmod}'),
      changeFrequency: '${e.changefreq}',
      priority: ${e.priority},
    }`).join(',\n')}
  ]
}
`
  const outputPath = path.join(appDir, 'sitemap.ts')
  fs.writeFileSync(outputPath, content)
  console.log(`✅ Next.js sitemap generated: ${outputPath}`)
}

main().catch(console.error)
