#!/usr/bin/env npx ts-node

/**
 * Bulk Page Generator for Programmatic SEO
 *
 * Usage:
 *   npx ts-node generate-pages.ts --input data.csv --template city-guide --output app/[city]
 *   npx ts-node generate-pages.ts --input data.json --template product-list --dry-run
 */

import * as fs from 'fs'
import * as path from 'path'

// Types
interface PageData {
  slug: string
  title: string
  [key: string]: string | number | boolean | string[]
}

interface Template {
  name: string
  pageTemplate: string
  generateMetadata: (data: PageData) => string
  generatePage: (data: PageData) => string
}

// Utility functions
function slugify(text: string): string {
  return text
    .toLowerCase()
    .trim()
    .replace(/[^\w\s-]/g, '')
    .replace(/[\s_-]+/g, '-')
    .replace(/^-+|-+$/g, '')
}

function parseCSV(content: string): PageData[] {
  const lines = content.trim().split('\n')
  const headers = lines[0].split(',').map(h => h.trim())

  return lines.slice(1).map(line => {
    const values = line.split(',').map(v => v.trim())
    const data: PageData = { slug: '', title: '' }
    headers.forEach((header, i) => {
      data[header] = values[i] || ''
    })
    if (!data.slug && data.title) {
      data.slug = slugify(String(data.title))
    }
    return data
  })
}

function parseJSON(content: string): PageData[] {
  const parsed = JSON.parse(content)
  return Array.isArray(parsed) ? parsed : parsed.data || parsed.items || []
}

// Templates
const templates: Record<string, Template> = {
  'city-guide': {
    name: 'City Guide',
    pageTemplate: 'page.tsx',
    generateMetadata: (data) => `
export async function generateMetadata({ params }: Props): Promise<Metadata> {
  const { city } = await params
  return {
    title: \`Best Places in ${data.title} | Your Site\`,
    description: \`Discover top-rated spots in ${data.title}. Updated guide with reviews.\`,
    openGraph: {
      title: \`Best Places in ${data.title}\`,
      type: 'article',
    },
  }
}`,
    generatePage: (data) => `
import type { Metadata } from 'next'
import { notFound } from 'next/navigation'

interface Props {
  params: Promise<{ city: string }>
}

${templates['city-guide'].generateMetadata(data)}

export default async function CityPage({ params }: Props) {
  const { city } = await params
  // Fetch data for: ${data.slug}

  return (
    <main className="max-w-4xl mx-auto px-4 py-12">
      <h1 className="text-4xl font-bold mb-6">Best Places in ${data.title}</h1>
      {/* Add your content components here */}
    </main>
  )
}

export async function generateStaticParams() {
  // Return all city slugs
  return [{ city: '${data.slug}' }]
}
`
  },
  'product-list': {
    name: 'Product List',
    pageTemplate: 'page.tsx',
    generateMetadata: (data) => `
export async function generateMetadata({ params }: Props): Promise<Metadata> {
  const { slug } = await params
  return {
    title: \`Best ${data.title} (${new Date().getFullYear()}) | Your Site\`,
    description: \`Compare top ${data.title}. Expert reviews and buying guide.\`,
  }
}`,
    generatePage: (data) => `
import type { Metadata } from 'next'

interface Props {
  params: Promise<{ slug: string }>
}

${templates['product-list'].generateMetadata(data)}

export default async function ProductListPage({ params }: Props) {
  const { slug } = await params

  return (
    <main className="max-w-4xl mx-auto px-4 py-12">
      <h1 className="text-4xl font-bold mb-6">Best ${data.title}</h1>
      {/* Add product comparison components here */}
    </main>
  )
}
`
  }
}

// Main function
async function main() {
  const args = process.argv.slice(2)
  const inputIndex = args.indexOf('--input')
  const templateIndex = args.indexOf('--template')
  const outputIndex = args.indexOf('--output')
  const dryRun = args.includes('--dry-run')

  if (inputIndex === -1) {
    console.error('Usage: npx ts-node generate-pages.ts --input <file> --template <name> [--output <dir>] [--dry-run]')
    process.exit(1)
  }

  const inputFile = args[inputIndex + 1]
  const templateName = templateIndex !== -1 ? args[templateIndex + 1] : 'city-guide'
  const outputDir = outputIndex !== -1 ? args[outputIndex + 1] : 'app/generated'

  // Read input file
  const content = fs.readFileSync(inputFile, 'utf-8')
  const isJSON = inputFile.endsWith('.json')
  const data = isJSON ? parseJSON(content) : parseCSV(content)

  console.log(`\n📊 Found ${data.length} items to generate`)
  console.log(`📝 Using template: ${templateName}`)
  console.log(`📁 Output directory: ${outputDir}\n`)

  const template = templates[templateName]
  if (!template) {
    console.error(`Template "${templateName}" not found. Available: ${Object.keys(templates).join(', ')}`)
    process.exit(1)
  }

  // Generate pages
  for (const item of data) {
    const pageDir = path.join(outputDir, item.slug)
    const pageFile = path.join(pageDir, template.pageTemplate)
    const pageContent = template.generatePage(item)

    if (dryRun) {
      console.log(`[DRY RUN] Would create: ${pageFile}`)
      console.log(`  Title: ${item.title}`)
      console.log(`  Slug: ${item.slug}`)
    } else {
      fs.mkdirSync(pageDir, { recursive: true })
      fs.writeFileSync(pageFile, pageContent.trim())
      console.log(`✅ Created: ${pageFile}`)
    }
  }

  console.log(`\n🎉 Done! Generated ${data.length} pages.`)
}

main().catch(console.error)
