# URL Structure for pSEO

SEO-friendly routing patterns for Next.js.

## URL Principles

1. **Readable**: `/coffee-shops/austin` not `/c/123`
2. **Hierarchical**: `/category/subcategory/item`
3. **Consistent**: Same pattern across all pages
4. **Short**: Remove stop words, use hyphens

## Common Patterns

| Page Type | Pattern | Example |
|-----------|---------|---------|
| City | `/[city]` | `/austin` |
| Category + City | `/[category]/[city]` | `/coffee-shops/austin` |
| State + City | `/[state]/[city]` | `/texas/austin` |
| Nested | `/[category]/[subcategory]/[slug]` | `/restaurants/italian/best-pizza` |

## Slugify Function

```typescript
export function slugify(text: string): string {
  return text
    .toLowerCase()
    .trim()
    .replace(/[^\w\s-]/g, '')     // Remove special chars
    .replace(/[\s_-]+/g, '-')     // Replace spaces with hyphens
    .replace(/^-+|-+$/g, '')      // Trim hyphens
}

// Usage
slugify("Best Coffee Shops!") // "best-coffee-shops"
slugify("Austin, TX")          // "austin-tx"
```

## Next.js Route Structure

```
app/
├── [city]/
│   ├── page.tsx              # /austin
│   └── [category]/
│       └── page.tsx          # /austin/restaurants
├── [state]/
│   └── [city]/
│       └── page.tsx          # /texas/austin
└── blog/
    └── [slug]/
        └── page.tsx          # /blog/best-coffee-austin
```

## Route Params Validation

```typescript
// lib/params.ts
const validCities = ['austin', 'dallas', 'houston']
const validCategories = ['restaurants', 'coffee-shops', 'hotels']

export function validateParams(city: string, category?: string): boolean {
  if (!validCities.includes(city)) return false
  if (category && !validCategories.includes(category)) return false
  return true
}

// In page.tsx
export default async function Page({ params }) {
  const { city } = await params
  if (!validateParams(city)) notFound()
  // ...
}
```

## generateStaticParams

```typescript
export async function generateStaticParams() {
  const cities = await getAllCities()
  const categories = await getAllCategories()

  const params = []
  for (const city of cities) {
    for (const category of categories) {
      params.push({ city: city.slug, category: category.slug })
    }
  }
  return params
}
```

## Trailing Slash Config

```typescript
// next.config.ts
const nextConfig = {
  trailingSlash: false,  // /austin not /austin/
}
```

## Redirects for Old URLs

```typescript
// next.config.ts
async redirects() {
  return [
    { source: '/old-url', destination: '/new-url', permanent: true },
    { source: '/city/:slug', destination: '/:slug', permanent: true },
  ]
}
```
