# Next.js Metadata for pSEO

Dynamic metadata generation for bulk pages.

## generateMetadata Function

```typescript
// app/[city]/page.tsx
import type { Metadata } from 'next'

interface Props {
  params: Promise<{ city: string }>
}

export async function generateMetadata({ params }: Props): Promise<Metadata> {
  const { city } = await params
  const data = await getCityData(city)

  return {
    title: `Best ${data.category} in ${data.name} | ${SITE_NAME}`,
    description: `Discover top-rated ${data.category} in ${data.name}, ${data.state}. Updated ${new Date().toLocaleDateString()}.`,
    openGraph: {
      title: `${data.count}+ Best ${data.category} in ${data.name}`,
      description: `Find the best ${data.category} in ${data.name}`,
      images: [`/og/${city}.png`],
      type: 'article',
    },
    alternates: {
      canonical: `https://example.com/${city}`,
    },
  }
}
```

## Title Patterns

```typescript
const titlePatterns = {
  'city-guide': '{count} Best {category} in {city} ({year})',
  'product-list': 'Best {category} - Top {count} Picks ({year})',
  'how-to': 'How to {action}: Complete Guide ({year})',
  'comparison': '{productA} vs {productB}: Which is Better?',
}

function generateTitle(template: string, data: Record<string, string>): string {
  return titlePatterns[template].replace(/{(\w+)}/g, (_, key) => data[key] || '')
}
```

## Description Patterns

```typescript
const descPatterns = {
  'city-guide': 'Discover the best {category} in {city}, {state}. Our curated list includes {count} top-rated spots with reviews and directions.',
  'product-list': 'Compare the {count} best {category} of {year}. Expert reviews, prices, and buying guide included.',
}
```

## Dynamic OG Images

```typescript
// app/og/[city]/route.tsx
import { ImageResponse } from 'next/og'

export async function GET(request: Request, { params }: { params: { city: string } }) {
  const data = await getCityData(params.city)

  return new ImageResponse(
    <div style={{ display: 'flex', flexDirection: 'column' }}>
      <h1>{data.title}</h1>
      <p>{data.count} places reviewed</p>
    </div>,
    { width: 1200, height: 630 }
  )
}
```

## robots.txt and Indexing

```typescript
// app/[city]/page.tsx
export const metadata: Metadata = {
  robots: {
    index: true,
    follow: true,
    googleBot: {
      index: true,
      follow: true,
      'max-image-preview': 'large',
      'max-snippet': -1,
    },
  },
}
```

## Canonical URLs

```typescript
function getCanonical(path: string): string {
  const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || 'https://example.com'
  return `${baseUrl}${path}`
}

// In generateMetadata
alternates: {
  canonical: getCanonical(`/${city}`),
  languages: {
    'en-US': getCanonical(`/en/${city}`),
    'es': getCanonical(`/es/${city}`),
  },
}
```
