# Keyword Clustering for pSEO

Group keywords by search intent to create focused, high-ranking pages.

## Clustering Strategy

### By Intent Type
| Intent | Example Keywords | Page Type |
|--------|-----------------|-----------|
| Informational | "how to", "what is", "guide" | Blog/Guide |
| Transactional | "buy", "price", "best" | Product/Landing |
| Navigational | "[brand] + [product]" | Brand pages |
| Local | "[service] in [city]" | Location pages |

### By Topic Modifier
```
Base keyword: "coffee shops"
Modifiers: [city], [neighborhood], [attribute]

Clusters:
- coffee shops in {city} → Location pages
- best coffee shops {city} → Listicle pages
- {attribute} coffee shops → Category pages
```

## Data Structure for Clustering

```typescript
interface KeywordCluster {
  primary: string           // Main keyword
  secondary: string[]       // Related terms
  intent: 'info' | 'trans' | 'nav' | 'local'
  volume: number           // Monthly search volume
  difficulty: number       // 0-100 scale
  template: string         // Which template to use
}

// Example cluster
const cluster: KeywordCluster = {
  primary: "coffee shops in austin",
  secondary: ["austin cafes", "best coffee austin tx"],
  intent: "local",
  volume: 2400,
  difficulty: 35,
  template: "city-guide"
}
```

## CSV Format for Bulk Import

```csv
keyword,intent,volume,difficulty,template,modifier_type,modifier_value
coffee shops in {city},local,2400,35,city-guide,city,austin
best {cuisine} restaurants,trans,1800,42,restaurant-list,cuisine,italian
```

## Clustering Tools

1. **Manual**: Export from Ahrefs/SEMrush → Group in spreadsheet
2. **Python**: Use `scripts/analyze-keywords.ts` with TF-IDF clustering
3. **API**: KeywordInsights, Cluster.ai for automated grouping

## Priority Matrix

```
High Volume + Low Difficulty = Priority 1 (Quick wins)
High Volume + High Difficulty = Priority 2 (Long-term)
Low Volume + Low Difficulty = Priority 3 (Fill gaps)
Low Volume + High Difficulty = Priority 4 (Skip)
```

## Output for Page Generation

After clustering, export to JSON for `generate-pages.ts`:
```json
{
  "clusters": [
    {
      "slug": "coffee-shops-austin",
      "title": "Best Coffee Shops in Austin",
      "keywords": ["coffee shops austin", "austin cafes"],
      "template": "city-guide",
      "data": { "city": "Austin", "state": "TX" }
    }
  ]
}
```
