# Internal Linking for pSEO

Programmatic link building between pages.

## Linking Strategies

### 1. Sibling Links (Same Level)
```typescript
// Link to related cities in same state
function RelatedCities({ state, currentCity }: { state: string; currentCity: string }) {
  const cities = getCitiesByState(state).filter(c => c.slug !== currentCity)
  return (
    <nav>
      <h2>More in {state}</h2>
      {cities.slice(0, 5).map(city => (
        <Link key={city.slug} href={`/${city.slug}`}>{city.name}</Link>
      ))}
    </nav>
  )
}
```

### 2. Parent/Child Links (Hierarchy)
```typescript
// Breadcrumb navigation
function Breadcrumbs({ items }: { items: Array<{ name: string; href: string }> }) {
  return (
    <nav aria-label="Breadcrumb">
      <ol>
        {items.map((item, i) => (
          <li key={item.href}>
            {i < items.length - 1 ? (
              <Link href={item.href}>{item.name}</Link>
            ) : (
              <span>{item.name}</span>
            )}
          </li>
        ))}
      </ol>
    </nav>
  )
}
```

### 3. Cross-Category Links
```typescript
// Link between different content types
function RelatedContent({ city, currentCategory }: Props) {
  const categories = ['restaurants', 'hotels', 'attractions']
    .filter(c => c !== currentCategory)

  return (
    <aside>
      <h3>Also in {city}</h3>
      {categories.map(cat => (
        <Link key={cat} href={`/${city}/${cat}`}>
          {capitalize(cat)} in {city}
        </Link>
      ))}
    </aside>
  )
}
```

## Link Component with Analytics

```typescript
function TrackedLink({ href, children, category }: Props) {
  return (
    <Link
      href={href}
      onClick={() => trackClick({ type: 'internal', href, category })}
    >
      {children}
    </Link>
  )
}
```

## Contextual In-Content Links

```typescript
// Auto-link mentions of cities in content
function AutoLinkedContent({ content, cities }: { content: string; cities: City[] }) {
  let linked = content
  for (const city of cities) {
    const regex = new RegExp(`\\b${city.name}\\b`, 'gi')
    linked = linked.replace(regex, `<a href="/${city.slug}">${city.name}</a>`)
  }
  return <div dangerouslySetInnerHTML={{ __html: linked }} />
}
```

## Link Density Guidelines

- **Min**: 2-3 internal links per 500 words
- **Max**: 1 link per 100 words (avoid over-linking)
- **Placement**: Early in content (first 200 words)
- **Anchor text**: Descriptive, keyword-rich

## Hub Page Pattern

```typescript
// Hub page linking to all child pages
function CategoryHub({ category, items }: Props) {
  return (
    <main>
      <h1>All {category}</h1>
      <nav>
        {items.map(item => (
          <Link key={item.slug} href={`/${category}/${item.slug}`}>
            <h2>{item.title}</h2>
            <p>{item.description}</p>
          </Link>
        ))}
      </nav>
    </main>
  )
}
```

## Footer Links Matrix

```typescript
function FooterLinks({ states, categories }: Props) {
  return (
    <footer>
      <div>
        <h4>By State</h4>
        {states.map(s => <Link key={s} href={`/state/${s}`}>{s}</Link>)}
      </div>
      <div>
        <h4>By Category</h4>
        {categories.map(c => <Link key={c} href={`/${c}`}>{c}</Link>)}
      </div>
    </footer>
  )
}
```
