# Content Templates for pSEO

Reusable page structures for consistent, scalable content.

## Template Architecture

```
components/
├── templates/
│   ├── city-guide.tsx      # Location-based pages
│   ├── product-list.tsx    # Comparison/listicle
│   ├── how-to-guide.tsx    # Tutorial pages
│   └── faq-page.tsx        # FAQ collections
└── blocks/
    ├── hero-section.tsx
    ├── feature-grid.tsx
    ├── cta-banner.tsx
    └── related-links.tsx
```

## City Guide Template

```typescript
// components/templates/city-guide.tsx
interface CityGuideProps {
  city: string
  state: string
  items: Array<{ name: string; rating: number; address: string }>
  faqs: Array<{ q: string; a: string }>
}

export function CityGuideTemplate({ city, state, items, faqs }: CityGuideProps) {
  return (
    <article>
      <HeroSection title={`Best Places in ${city}, ${state}`} />
      <IntroSection city={city} count={items.length} />
      <ItemList items={items} />
      <FAQSection faqs={faqs} />
      <RelatedCities state={state} currentCity={city} />
    </article>
  )
}
```

## Product List Template

```typescript
interface ProductListProps {
  category: string
  products: Array<{
    name: string
    price: number
    rating: number
    features: string[]
  }>
}

export function ProductListTemplate({ category, products }: ProductListProps) {
  return (
    <article>
      <h1>Best {category} in 2024</h1>
      <QuickPicks products={products.slice(0, 3)} />
      <ComparisonTable products={products} />
      <DetailedReviews products={products} />
      <BuyingGuide category={category} />
    </article>
  )
}
```

## Content Blocks

### Dynamic Intro
```typescript
function IntroSection({ city, count }: { city: string; count: number }) {
  const intros = [
    `Discover the top ${count} spots in ${city}.`,
    `Looking for the best in ${city}? We've got you covered.`,
    `Our guide to ${city}'s finest ${count} locations.`
  ]
  return <p>{intros[hash(city) % intros.length]}</p>
}
```

### Related Links Block
```typescript
function RelatedLinks({ items }: { items: Array<{ href: string; title: string }> }) {
  return (
    <nav aria-label="Related pages">
      <h2>You might also like</h2>
      <ul>{items.map(item => <li key={item.href}><Link href={item.href}>{item.title}</Link></li>)}</ul>
    </nav>
  )
}
```

## Template Selection Logic

```typescript
function getTemplate(cluster: KeywordCluster): ComponentType {
  const templates = {
    'city-guide': CityGuideTemplate,
    'product-list': ProductListTemplate,
    'how-to': HowToTemplate,
    'faq': FAQTemplate
  }
  return templates[cluster.template] || DefaultTemplate
}
```
