---
name: programmatic-seo
description: Programmatic SEO for Next.js. Use when building content at scale, keyword clustering, bulk page generation, dynamic routes with SEO optimization, sitemap generation, or schema markup. Triggers on requests involving SEO strategy, content templates, metadata patterns, or scaling content production.
version: 1.0.0
---

# Programmatic SEO Expert

Build SEO-optimized content at scale using Next.js dynamic routes, templates, and automation.

## pSEO Workflow

```
1. Keyword Research → Cluster keywords by intent/topic
2. Data Source → Prepare structured data (CSV/JSON/CMS)
3. Templates → Create reusable content components
4. Dynamic Routes → Generate pages from data
5. Technical SEO → Metadata, schema, sitemap, internal links
```

## Quick Start

### Bulk Page Generation
Run `scripts/generate-pages.ts` to scaffold pages from CSV/JSON data:
```bash
npx ts-node scripts/generate-pages.ts --input data/keywords.csv --template city-guide
```

### Sitemap Generation
Run `scripts/generate-sitemap.ts` for dynamic sitemap:
```bash
npx ts-node scripts/generate-sitemap.ts --base-url https://example.com
```

## References

| Topic | File | Use When |
|-------|------|----------|
| Keyword Clustering | `references/keyword-clustering.md` | Grouping keywords by intent |
| Content Templates | `references/content-templates.md` | Creating reusable page structures |
| Next.js Metadata | `references/nextjs-metadata.md` | Dynamic meta tags, OG images |
| Schema Markup | `references/schema-markup.md` | Structured data for rich results |
| URL Structure | `references/url-structure.md` | SEO-friendly routing patterns |
| Internal Linking | `references/internal-linking.md` | Programmatic link strategies |

## Core Patterns

### Dynamic Route Structure
```
app/
├── [city]/
│   └── page.tsx           # /new-york, /los-angeles
├── [category]/[slug]/
│   └── page.tsx           # /restaurants/best-pizza
└── sitemap.ts             # Dynamic sitemap
```

### Data-Driven Content
```typescript
// lib/data.ts - Load from CSV/JSON/CMS
export async function getAllCities() {
  return data.map(row => ({
    slug: slugify(row.city),
    title: row.city,
    state: row.state,
    population: row.population
  }))
}
```

### generateStaticParams Pattern
```typescript
export async function generateStaticParams() {
  const cities = await getAllCities()
  return cities.map(city => ({ city: city.slug }))
}
```

## Scripts

| Script | Purpose |
|--------|---------|
| `scripts/generate-pages.ts` | Scaffold bulk pages from data |
| `scripts/generate-sitemap.ts` | Create dynamic XML sitemap |
| `scripts/analyze-keywords.ts` | Cluster keywords by intent |

## SEO Checklist

- [ ] Unique title/description per page
- [ ] Schema markup (Article, LocalBusiness, FAQ)
- [ ] Canonical URLs
- [ ] Dynamic sitemap with lastmod
- [ ] Internal linking between related pages
- [ ] Image optimization with alt text
- [ ] Core Web Vitals optimized
